/***************************************************************************
 *   Copyright (C) 2006 by Emanuel Wegh                                    *
 *   maan@ddsw.nl                                                          *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include <QtGui>

#include "robotgui.h"
#include "rsocket.h"
#include "bulb.h"
#include "camerawidget.h"
#include "qsliderarrow.h"

RobotGui::RobotGui(QWidget *parent) : QDialog(parent)
{
	setWindowTitle(tr("Robot Client"));
	
	// Robot Control
	QGroupBox *steeringGroup = new QGroupBox(tr("Steering Control"));

	horizontalSlider = new QSliderArrow(Qt::Horizontal, steeringGroup);
	horizontalSlider->setMinimum(-20);
	horizontalSlider->setMaximum(20);
	horizontalSlider->setTickPosition(QSlider::TicksBelow);
	horizontalSlider->setTickInterval(10);
	horizontalSlider->setToolTip(tr("Steering"));

	verticalSlider = new QSliderArrow(Qt::Vertical, steeringGroup);
	verticalSlider->setMinimum(-100);
	verticalSlider->setMaximum(100);
	verticalSlider->setTickPosition(QSlider::TicksBothSides);
	verticalSlider->setTickInterval(10);
	verticalSlider->setMinimumHeight(130);
	verticalSlider->setToolTip(tr("Speed"));

	QPushButton *breakButton = new QPushButton(tr("Stop"));
	breakButton->setPalette(Qt::red);
	breakButton->setToolTip(tr("Stop Robot"));

	QPushButton *neutralButton = new QPushButton(tr("Straight Ahead"));
	neutralButton->setToolTip(tr("Drive straight ahead"));
	
	QCheckBox *normalSteeringCheckBox = new QCheckBox(tr("Normal Steering"));
	normalSteeringCheckBox->setToolTip(tr("Fast steering is also possible"));
	normalSteeringCheckBox->setCheckState(Qt::Checked);

	connect(breakButton, SIGNAL(clicked()), this, SLOT(slowBreak()));
	connect(neutralButton, SIGNAL(clicked()), this, SLOT(straightAhead()));
	connect(normalSteeringCheckBox, SIGNAL(stateChanged(int)), this, SLOT(steeringSpeed(int)));


	// Connection
	QPushButton *connectButton = new QPushButton(tr("Connect"));
	connectButton->setToolTip(tr("Connect to robot server"));
	
	QLabel *connectLabel = new QLabel("Connection:");
	connectLabel->setMaximumHeight(25);
	
	hostName = new QLineEdit();
	hostName->setToolTip(tr("Type robot servername"));
	connectBulb = new Bulb();
	connectBulb->setToolTip(tr("Connection status"));
	
	
	// Camera
	cameraWidget = new CameraWidget();

	    
	// Other
	QPushButton *quitButton = new QPushButton(tr("Quit"));
	quitButton->setToolTip(tr("Quit application"));
	
	QPushButton *infoButton = new QPushButton(tr("Info"));
	infoButton->setToolTip(tr("Application info"));

	connect(quitButton, SIGNAL(clicked()), this, SLOT(close()));
	connect(infoButton, SIGNAL(clicked()), this, SLOT(info()));


	// Logging in Textbox
	textBox = new QTextEdit();
	textBox->setReadOnly(true);
	textBox->setMaximumHeight(100);
	textBox->setAlignment(Qt::AlignLeft);
	textBox->append("Robot client started");


	// Layouts
	QHBoxLayout *cameraLayout = new QHBoxLayout();
	cameraLayout->addWidget(cameraWidget, 0, Qt::AlignHCenter);
	
	QVBoxLayout *sliderLayout = new QVBoxLayout(steeringGroup);
	sliderLayout->addWidget(verticalSlider, 0, Qt::AlignHCenter);
	sliderLayout->addWidget(horizontalSlider);
	steeringGroup->setLayout(sliderLayout);
	
	QHBoxLayout *connectLayout = new QHBoxLayout();
	connectLayout->addStretch(1);
	connectLayout->addWidget(connectLabel);
	connectLayout->addWidget(connectBulb);
    
	QGridLayout *gridLayout = new QGridLayout();
	gridLayout->addWidget(hostName, 1, 1);
	gridLayout->addWidget(connectButton, 1, 2);
	gridLayout->addLayout(connectLayout, 2, 1);
	gridLayout->setRowStretch(3, 1);
	gridLayout->addWidget(normalSteeringCheckBox, 4, 1);
	gridLayout->addWidget(breakButton, 5, 1);
	gridLayout->addWidget(neutralButton, 6, 1);
	gridLayout->addWidget(infoButton, 5, 2);
	gridLayout->addWidget(quitButton, 6, 2);

	QHBoxLayout *mainLayout = new QHBoxLayout();
	mainLayout->addWidget(steeringGroup);
	mainLayout->addLayout(gridLayout);

	QVBoxLayout *layout = new QVBoxLayout(this);
	layout->addLayout(cameraLayout);
	layout->addLayout(mainLayout);
	layout->addWidget(textBox);

	setLayout(layout);


	// Init communication with robot server
	rSocket = new RSocket(cameraWidget, this);

	// Init communication connections
	connect(connectButton, SIGNAL(clicked()), this, SLOT(connectToHost()));
	connect(verticalSlider, SIGNAL(valueChanged(int)), rSocket, SLOT(changeSpeed(int)));
	connect(horizontalSlider, SIGNAL(valueChanged(int)), rSocket, SLOT(changeSteering(int)));

	connect(rSocket, SIGNAL(connected()), this, SLOT(connectionEstablished()));
	connect(rSocket, SIGNAL(connected()), this, SLOT(slowBreak()));    
	connect(rSocket, SIGNAL(error(QAbstractSocket::SocketError)),
    	    this, SLOT(connectionError(QAbstractSocket::SocketError)));
}

void RobotGui::slowBreak()
{
	horizontalSlider->setValue(0);
	verticalSlider->setValue(0);
}

void RobotGui::straightAhead()
{
	horizontalSlider->setValue(0);
}

void RobotGui::steeringSpeed(int state)
{
	if (state == Qt::Checked)
		horizontalSlider->setRange(-20, 20);
	if (state == Qt::Unchecked)
		horizontalSlider->setRange(-100, 100);
}


void RobotGui::connectToHost()
{
	// Abort current connection
	rSocket->abort();
	connectBulb->redBulb();

	textBox->append("Connecting to " + hostName->text());
    
	// Connect
	rSocket->hostName = hostName->text();
	rSocket->connectToHost();
}

void RobotGui::connectionEstablished()
{
	textBox->append("Connected to robot server");
	connectBulb->greenBulb();
}

void RobotGui::connectionError(QAbstractSocket::SocketError socketError)
{
	textBox->append(rSocket->displayError(socketError));
	connectBulb->redBulb();
}


void RobotGui::info()
{
	QMessageBox::information(this, ROBOT_INFO,
		ROBOT_INFO
		"\n\n"
		"This application has been made to control a robot, which you can find at:\n"
		"http://maan.hijmans.nl/robot/\n\n"
		"License information:\n"
		"This program is distributed under the terms of the GPL v2.\n"
		"http://www.gnu.org/licenses/\n\n"
		"(c) 2006, Emanuel Wegh (maan@ddsw.nl)");
}
