/***************************************************************************
 *   Copyright (C) 2006 by Emanuel Wegh                                    *
 *   maan@ddsw.nl                                                          *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include <QtGui>
#include <QtNetwork>

#include "rsocket.h"
#include "camerawidget.h"

RSocket::RSocket(CameraWidget *camera, QObject *parent) : QTcpSocket(parent)
{
	// init
	commSelector = NO_SELECTOR;
	bufferSizeRead = false;
	heartBeat = new QTimer(this);
	imageBuffer = new QByteArray();
	cameraWidget = camera;

	// init heartbeat
	connect(this, SIGNAL(readyRead()), this, SLOT(readServerData()));
	connect(heartBeat, SIGNAL(timeout()), this, SLOT(sendHeartBeat()));  
	connect(this, SIGNAL(disconnected()), this, SLOT(stopHeartBeat()));
}

RSocket::~RSocket()
{
	if (this->state() == QAbstractSocket::ConnectedState)
		disconnectFromHost();
	
	delete(imageBuffer);
}

void RSocket::readServerData()
{
	QDataStream buffer(this);

	// Read communication selector
	if (commSelector == NO_SELECTOR)
		if (bytesAvailable() < sizeof(qint8))
			return;
		else
			buffer >> commSelector;

	
	// The robot server already serves a client. Generate an error!
	if (commSelector == ALREADY_CONNECTED)
	{
		emit error(QAbstractSocket::UnknownSocketError);
		
		commSelector = NO_SELECTOR;
	}
	
			
	// Read / write first information to initialize communication session
	if (commSelector == ROBOT_VERSION_INFO)
	{
		// Start new communication session with robot server.

		if (bytesAvailable() < sizeof(float))
			return;
		else
			buffer >> robotServerVersion;	// Read server version
	
		// controleer server versie !!!!!!!!!!!!!!!!!!!!
	
		// Write robot client version
		buffer << ROBOT_VERSION_INFO;		// communication selector
		buffer << ROBOT_VERSION;
	
		// Write robot communication ID
		buffer << COMM_ID_INFO;				// communication selector
		buffer << COMM_ID;
	
		// Communication session is now initialized
	       
		// Start sending heartbeat
		heartBeat->start(HEART_BEAT_PERIOD);
	
		commSelector = NO_SELECTOR;
	}

	if (commSelector == SELECT_IMAGE)
	{
		static int bufferSize;
	
		// First read image buffer size
		if (!bufferSizeRead)
			if (bytesAvailable() < sizeof(qint32))
				return;
			else
			{
				buffer >> bufferSize;
				bufferSizeRead = true;
			}
			
		// Read jpeg image from server
		if (bytesAvailable() < bufferSize)
			return;
	
		imageBuffer->resize(bufferSize);
		*imageBuffer = read(bufferSize);
		
		// Write image to screen
		cameraWidget->setImage(imageBuffer);
	
		bufferSizeRead = false;	
		commSelector = NO_SELECTOR;
	}
}

void RSocket::connectToHost()
{
	// First close or abort current connection
	if (state() == QAbstractSocket::ConnectedState)
		close();
	if (state() == QAbstractSocket::HostLookupState ||
		state() == QAbstractSocket::ConnectingState)
		abort(); 

	// New connection  
	QTcpSocket::connectToHost(hostName, TCP_PORT);
}

void RSocket::changeSpeed(int speed)
{
	QByteArray block;
	QDataStream buffer(&block, QIODevice::WriteOnly); 

	buffer << SELECT_SPEED;	// communication selector
	buffer << speed;

	write(block);
}

void RSocket::changeSteering(int steering)
{
	QByteArray block;
	QDataStream buffer(&block, QIODevice::WriteOnly);

	buffer << SELECT_STEERING;	// communication selector
	buffer << steering;

	write(block);
}

void RSocket::sendHeartBeat()
{
	QDataStream buffer(this); 

	buffer << HEART_BEAT;	// communication selector
}

void RSocket::stopHeartBeat()
{
	heartBeat->stop();
}

QString RSocket::displayError(SocketError socketError)
{
	// Connection error messages
	QString errorMessage;

	switch (socketError)
	{
		case QAbstractSocket::RemoteHostClosedError:
			errorMessage = "Host closed connection";
			break;
		case QAbstractSocket::HostNotFoundError:
			errorMessage = "The host was not found";
			break;
		case QAbstractSocket::ConnectionRefusedError:
			errorMessage = "The connection was refused by the peer";
			break;
		case QAbstractSocket::UnknownSocketError:
			errorMessage = "A client is already connected to the robot";
			break;
		default:
			errorMessage = errorString();
	}

	return errorMessage;
}
