 /***************************************************************************
 *   Copyright (C) 2006 by Emanuel Wegh                                    *
 *   maan@ddsw.nl                                                          *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include "image.h"


Image::Image(int imageWidth, int imageHeight)
{
	// init
	width  = imageWidth;
	height = imageHeight;
	pixels = width * height;
	imageSize = pixels * 3;		// 3 bytes/pixel

	// Create image buffer
	imageBuffer = new unsigned char[imageSize];
}

Image::~Image()
{
	delete(imageBuffer);
}

float Image::highExposedPixels()
{
	// Weighted portion of highly exposed (value > 255-64) pixels.
	// Highest value 255 counts as: 1
	// Lowest value 255-64 counts as: 0

	int i;
	unsigned char *image;
	float highExposure = 0;
	
	image = imageBuffer + 1;				// select green channel

	for(i = 0; i < pixels; i++)
	{
		// Only use high exposed pixels
		//if (meanRGB(image) > 255 - 64)	// use RGB
		if (*image > 255 - 64)				// use green channel only (faster)
			highExposure += (float)(*image - 255 + 64) / 64;
		image += 3;
	}

	return highExposure / pixels;
}

float Image::lowExposedPixels()
{
	// Weighted portion of low exposed (value < 64) pixels.
	// Lowest value 0 counts as: 1
	// Highest value 64 counts as: 0

	int i;
	unsigned char *image;
	float lowExposure = 0;

	image = imageBuffer + 1;				// select green channel

	for(i = 0; i < pixels; i++)
	{
		// Only use low exposed pixels
		//if (meanRGB(image) < 64)			// use RGB
		if (*image < 64)					// use green channel only (faster)
			lowExposure += (float)(64 - *image) / 64;
		image += 3;
	}

	return lowExposure / pixels;
}

void Image::swapRedBlueChannel()
{
	int i;
	unsigned char *image, red;
	
	image = imageBuffer;
	
	for(i = 0; i < pixels; i++)
	{
		red = *image;			// remember red pixel
		*image = *(image + 2);	// write blue pixel to red pixel
		*(image + 2) = red;		// write red pixel to blue pixel
		
		image += 3;
	}
}

void Image::changeBlueIntensity(float intensity)
{
	// This function is much slower then it's int variant below.
	
	int i;
	unsigned char *image;
	float blue;
	
	image = imageBuffer + 2;
	
	for(i = 0; i < pixels; i++)
	{
		blue = (float)*image;
		blue *= intensity;
		
		// Check min/max values
		if (blue > 255)
			blue = 255;
		//if (blue < 0)		not in use with only positive intensity values !!!!!!!!
		//	blue = 0;
		
		// Write new blue value
		*image = (unsigned char)blue;
		
		image += 3;
	}
}

void Image::changeBlueIntensity(int intensity)
{
	// 'intensity' is an int multiplication factor multiplied by 100.
	// It will be divided by 100 after the multiplication.
	// This is much faster then a floating point multiplication.
	
	int i, blue;
	unsigned char *image;
	
	image = imageBuffer + 2;
	
	for(i = 0; i < pixels; i++)
	{
		blue = (int)*image;
		blue *= intensity;
		blue /= 100;
		
		// Check min/max values
		if (blue > 255)
			blue = 255;
		//if (blue < 0)		not in use with only positive intensity values !!!!!!!!
		//	blue = 0;
		
		// Write new blue value
		*image = (unsigned char)blue;
		
		image += 3;
	}
}

unsigned char Image::meanRGB(const unsigned char *pixel)
{
	// Calculates mean value of the RGB colors of a pixel.
	// This gives its gray value.

	return (*pixel + *(pixel + 1) + *(pixel + 2)) / 3;
}

