 /***************************************************************************
 *   Copyright (C) 2006 by Emanuel Wegh                                    *
 *   maan@ddsw.nl                                                          *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include <QtCore> 

#include <stdlib.h>
#include <stdio.h>
#include <math.h>
#include <time.h>

#include "robotcontrol.h" 
#include "TeleoEasy.h"
#include "TeleoIntroPwm.h"
#include "TeleoIntroDout.h"
#include "TeleoIntroDin.h"
#include "TeleoIntroAin.h"
#include "TeleoMotor.h"


RobotControl* RobotControl::ptr = 0;

RobotControl::RobotControl(QObject *parent) : QObject(parent)
{
	// init
	ptr = this;				// used by wrapper function
	externalPower = false;	// pc starts default on battery power (hardware default)
	
	// This timer is used to slow down changes in motor speed.
	timer = new QTimer(this);
	connect(timer, SIGNAL(timeout()), this, SLOT(changeMotorSpeedSlowly()));
	
	// This timer delays external power check.
	extPowerTimer = new QTimer(this);
	connect(extPowerTimer, SIGNAL(timeout()), this, SLOT(checkExtPowerStatus()));
}

RobotControl::~RobotControl()
{
	// Stop al robot activity before destroying this object.
	stopMotors();
	turnOffConnectLed();
}


bool RobotControl::startTeleo()
{
	// init
	TeleoDeviceManager *tdm;

	// Connect teleo
	if (TeleoEasy_Init(NULL, TI_USB, &tdm) != TELEO_OK)
		return false;

	// Create teleo proxies
	//if (TIntroPwm_Create(tdm, NULL, 1, NULL, &light) != TELEO_OK)
	if (TIntroDout_Create(tdm, NULL, 0, NULL, &lowPowerLed) != TELEO_OK)
		return false;
	if (TIntroDout_Create(tdm, NULL, 1, NULL, &connectLed) != TELEO_OK)
		return false;
	if (TIntroPwm_Create(tdm, NULL, 1, NULL, &extPowerSwitch) != TELEO_OK)
		return false;
	if (TIntroDin_Create(tdm, NULL, 0, NULL, &extPower) != TELEO_OK)
		return false;
	if (TIntroAin_Create(tdm, NULL, 0, NULL, &pcV) != TELEO_OK)
		return false;
	if (TIntroAin_Create(tdm, NULL, 1, NULL, &motorV) != TELEO_OK)
		return false;
	// Use device address instead of NULL to bound a proxy.

	// Bounding with NULL takes longer and generates errors while initializing the motors.
	if (TMotor_Dual2ACreate(tdm, "6", 0, NULL, &motorLeft) != TELEO_OK)
		return false;
	if (TMotor_Dual2ACreate(tdm, "6", 1, NULL, &motorRight) != TELEO_OK)
		return false;

	// Start Teleo
	TeleoEasy_Start();

	// Set motor ranges & callback
	TeleoEasy_Lock();
	//teleoError = TIntroPwm_dutyMaxSet(light, 100);
	teleoError = TMotor_speedMaxSet(motorLeft, MAX_LOG_SPEED);
	teleoError = TMotor_speedMaxSet(motorRight, MAX_LOG_SPEED);
	teleoError = TIntroDin_valueCallbackSet(extPower, extPowerUpdateWrapper);
	teleoErrorHandler(teleoError, "extPower");
	teleoError = TIntroAin_valueCallbackSet(pcV, checkPCVoltageWrapper);
	teleoError = TIntroAin_resolutionSet(pcV, 10);
	teleoError = TIntroAin_samplePeriodSet(pcV, 2000);
	teleoError = TIntroAin_valueCallbackSet(motorV, checkTeleoVoltage);
	teleoError = TIntroAin_resolutionSet(motorV, 10);
	teleoError = TIntroAin_samplePeriodSet(motorV, 2000);
	maxPowerDutyValue = 10;
	teleoError = TIntroPwm_dutyMaxSet(extPowerSwitch, maxPowerDutyValue);		// this initializes powerswitch to accu power.
	// alleen een wijziging naar eeb digitalepoort kan dit voorkomen.
	TeleoEasy_Unlock();

	// Turn off connected devices
	wait(2);	// Use a delay of 2 sec. Teleo has to initialize first.
	stopMotors();
	turnOffConnectLed();
	turnOffLowPowerLed();
	
	// Check if connected to external power. Use a delay of 2 sec. Teleo has to initialize first.
	//extPowerTimer->start(3000); 	werkt niet

	// Teleo initialized correctly
	return true;
}


void RobotControl::setSpeed(const int speed)
{
	// speed = current speed

	newSpeed = speed;
	checkMaxValue(newSpeed, MIN_SPEED, MAX_SPEED);
	startChangingMotorSpeed();
}

void RobotControl::setSteering(const int steering)
{
	// steering = current steering

	newSteering = steering;
	checkMaxValue(newSteering, MIN_STEERING, MAX_STEERING);
	startChangingMotorSpeed();
}


void RobotControl::startChangingMotorSpeed()
{
	// Motor speed is not changed immediately, but slowly by the timer.
	if (!timer->isActive())
		timer->start(100);	// change speed every 100 ms
}

void RobotControl::changeMotorSpeedSlowly()
{
	// Change (current) speed / steering slowly to new values

	int stepD2 = SPEED_STEP_SIZE / 2;

	// Check if current speed and steering are finished
	if (speed == newSpeed && steering == newSteering)
	{
		timer->stop();	// When current speed / steering is equal to new values: stop timer
		return;
	}

	// Change speed if necessary
	if (speed >= newSpeed - stepD2 && speed <= newSpeed + stepD2)
		speed = newSpeed;
	else
	{
		// slowly change current values to new values
		// don't change a value when already equal to new value
		if (speed < newSpeed)
				speed += SPEED_STEP_SIZE;
		else
			if (speed > newSpeed)
				speed -= SPEED_STEP_SIZE;
	}
     
	// Change steering if necessary 
	if (steering >= newSteering - stepD2 && steering <= newSteering + stepD2)
		steering = newSteering;
	else
	{
		// slowly change current values to new values
		// don't change a value when already equal to new value
		if (steering < newSteering)
				steering += SPEED_STEP_SIZE;
		else
			if (steering > newSteering)
				steering -= SPEED_STEP_SIZE;
	}
	
	// set new motor speed
	updateMotorSpeed();

}

void RobotControl::updateMotorSpeed()
{
	// Change (current) speed / steering immediately

	int motorSpeedLeft, motorSpeedRight;

	// Calculate left / right motor speed
	motorSpeedLeft  = speed - steering;
	motorSpeedRight = speed + steering;

	// Check and correct max. speed of motors
	checkMaxValue(motorSpeedLeft , MIN_SPEED, MAX_SPEED);
	checkMaxValue(motorSpeedRight, MIN_SPEED, MAX_SPEED);

	// Set motor speed
	TeleoEasy_Lock();
	teleoError1 = TMotor_speedSet(motorLeft , logSpeed(motorSpeedLeft));
	teleoError2 = TMotor_speedSet(motorRight, logSpeed(motorSpeedRight));
	TeleoEasy_Unlock();
	
	//printf("Motorspeed: %d, %d\n", motorSpeedLeft, motorSpeedRight);
	
	// Error handling
	if (teleoError1 != TELEO_OK)
		teleoErrorHandler(teleoError1, "Motor Left error");
	if (teleoError2 != TELEO_OK)
		teleoErrorHandler(teleoError2, "Motor Right error");
}

void RobotControl::stopMotors()
{
	// Stop the motors immediately

	speed = 0;
	steering = 0;

	// Stop timer when active
	if (timer->isActive())
		timer->stop();

	updateMotorSpeed();
}



// LED control
void RobotControl::turnOnLowPowerLed()
{
	//TeleoEasy_Lock();		dit blokkeerd bij aanroep checkPCVoltage via wrapper.
	teleoError = TIntroDout_valueSet(lowPowerLed, true);
	if (teleoError != TELEO_OK)
		teleoErrorHandler(teleoError, "Turn on low power led");
	//TeleoEasy_Unlock();
}

void RobotControl::turnOffLowPowerLed()
{
	TeleoEasy_Lock();
	teleoError = TIntroDout_valueSet(lowPowerLed, false);
	if (teleoError != TELEO_OK)
		teleoErrorHandler(teleoError, "Turn off low power led");
	TeleoEasy_Unlock();
	
}

void RobotControl::turnOnConnectLed()
{
	TeleoEasy_Lock();
	teleoError = TIntroDout_valueSet(connectLed, true);
	if (teleoError != TELEO_OK)
		teleoErrorHandler(teleoError, "Turn on connect led");
	TeleoEasy_Unlock();
}

void RobotControl::turnOffConnectLed()
{
	TeleoEasy_Lock();
	teleoError = TIntroDout_valueSet(connectLed, false);
	if (teleoError != TELEO_OK)
		teleoErrorHandler(teleoError, "Turn off connect led");
	TeleoEasy_Unlock();
}

void RobotControl::switchToExtPower()
{
	//TeleoEasy_Lock();		dit blokkeerd bij aanroep via wrapper.
	if (externalPower)
	{
		teleoError = TIntroPwm_dutySet(extPowerSwitch, maxPowerDutyValue);
		if (teleoError != TELEO_OK)
			teleoErrorHandler(teleoError, "Switch to external power");
	}

	//TeleoEasy_Unlock();
}

void RobotControl::switchToAccuPower()
{
	TeleoEasy_Lock();
	teleoError = TIntroPwm_dutySet(extPowerSwitch, 0);
	if (teleoError != TELEO_OK)
		teleoErrorHandler(teleoError, "Switch to accu power");
	TeleoEasy_Unlock();
}

bool RobotControl::getExtPowerStatus()
{
	return externalPower;
}

void RobotControl::checkExtPowerStatus()
{
	printf("Start checkExtPowerStatus\n");
	
	if (externalPower) 
	{
		printf("Switch to external power\n");
		switchToExtPower();
	}
	else
		printf("No external power switch\n");
	
	extPowerTimer->stop();
	
	printf("End checkExtPowerStatus\n");
}




// Wrapper to extPowerUpdate, used by Teleo C callback function.
TeleoError RobotControl::extPowerUpdateWrapper(TIntroDin *tmdin, bool power)
{
	return ptr->extPowerUpdate(tmdin, power);
}

TeleoError RobotControl::extPowerUpdate(TIntroDin *, bool power)
{
	printf("extPowerUpdate call\n");
	if (power && !externalPower)
	{
		printf("On Ext Power\n");
		
		// Start switching to external power procedure.
		// Check after 1 sec if still connected to external power. Then switch power.
		// This gives some time to connect the external power cable correctly.
		//extPowerTimer->start(5000);	// kan problemen via wrapper geven. werkt soms wel, soms niet.
		externalPower = power;		// needed by switchToExtPower as a power check
		sleep(1);
		checkExtPowerStatus();
		//switchToExtPower();
	}
	else
		printf("No Ext Power\n");
	
	externalPower = power;
		
	return TELEO_OK;
}

// Wrapper to checkPCVoltage, used by Teleo C callback function.
TeleoError RobotControl::checkPCVoltageWrapper(TIntroAin *tma, short int value)
{
	return ptr->checkPCVoltage(tma, value);
}

TeleoError RobotControl::checkPCVoltage(TIntroAin *, short int value)
{
	float volt;
	static QTime nextTime, curTime;
	static int lowPowerCounter = 0;
	FILE *fd;
		
	volt = value * 10.88 / 807;		// calculate voltage
		
	if (!externalPower)
	{
		if (volt < 11.0)
			turnOnLowPowerLed();
		// The first voltage measurement is not always accurate and can lead to a lowpower measurement,
		// which will shutdown the pc. This is not practical.
		// The lowPowerCounter is used to count a few lowpower measurements, which makes the lowpower status certain.
		if (volt < 10.8 & volt > 1.0)	// voltage lower than 1.0 V is a wrong measurement. Ignore!
			lowPowerCounter++;	
		else
			lowPowerCounter = 0;
		
		if (lowPowerCounter == 3)
		{
			// shutdown robot
			fd = fopen("/home/maan/robot/control", "w");
			fputc('1', fd);
			fclose(fd);
		}
	}
			
	// Write voltage to std out
	curTime = QTime::currentTime();
	if (curTime > nextTime)			// don't write too often
	{
		printf("Volt A: %d, %f, ", value, volt);
		printf(curTime.toString("hh:mm:ss").toAscii());
		printf("\n");
		nextTime = curTime.addSecs(60);
	}
	
	return TELEO_OK;
}

TeleoError RobotControl::checkTeleoVoltage(TIntroAin *, short int value)
{
	float volt;
	static QTime nextTime, curTime;
	
	volt = value * 10.92 / 819;		// calculate voltage
		
	// Write voltage to std out
	curTime = QTime::currentTime();
	if (curTime > nextTime)			// don't write too often
	{
		printf("Volt B: %d, %f, ", value, volt);
		printf(curTime.toString("hh:mm:ss").toAscii());
		printf("\n");
		nextTime = curTime.addSecs(60);
	}	
	
	return TELEO_OK;
}



void RobotControl::checkMaxValue(int &speed, const int min, const int max)
{
	if (speed > max)
		speed = max;
	if (speed < min)
		speed = min;
}

float RobotControl::logSpeed(const int speed)
{
	bool  negative = false;
	float newSpeed, lineairSpeed;

	lineairSpeed = (float)speed;

	// Log function can't work with negative values
	// Make them positive
	if (lineairSpeed < 0)
	{
		negative = true;
		lineairSpeed = -lineairSpeed;
	}

	// Calculate log speed
	// Log function needs 1 as a minimum: speed + 1
	newSpeed = log(lineairSpeed + 1) / log((float)MAX_SPEED + 1) * MAX_LOG_SPEED;

	// Don't forget negative values
	if (negative)
		newSpeed = -newSpeed;

	return(newSpeed);
}


void RobotControl::teleoErrorHandler(TeleoError error, char *errorText)
{
	// Write error to console
	printf("%s: Teleo Error: %d, %s\n", errorText, error, teleoErrorText[error]);
	
	// Teleo library loses regulary its connection with the hardware. Reason unknown.
	// Solution: restart Teleo system.
	//if (error == TELEO_E_BOUND)
	//{
	//	printf("Restart Teleo system:\n");
	/*
		// TeleoEasy_End only works when added to TeleoEasy.h !!!!!!!
		TeleoEasy_End();		// stop Teleo system
		printf("Teleo stopped succesfully\n");
			
		if(startTeleo())		// start Teleo system
			printf("Teleo restarted succesfully\n");
		else
			printf("Teleo restart failed\n");
	*/
	//}
}

void RobotControl::wait (int seconds)
{
	clock_t endwait;
	endwait = clock () + seconds * CLOCKS_PER_SEC ;
	while (clock() < endwait) {}
}
